 aR  w / m^9      h	 oP    nSystem-wide    NAME RomIntr

; This is RomIntr.Asm.  This file contains
; the routines that deal with the low level
; roms as access via the ROM BIOS interface

CGROUP GROUP CODE

PUBLIC RomSubsystem

; constants

numValidRomFncs   EQU 5
numLogHwSlots     EQU 8
numPhysHwSlots    EQU 8

romErrInvFnc      EQU 1	; Function requested is out of range
romErrInvDataBlk  EQU 2	; Data block passed in DS is invalid
romErrNoRomsExist EQU 3	; There are no (valid) ROMs installed
romErrInvImageNum EQU 4	; ROM image does not exist

true              EQU 1
false	         EQU 0

enableRom         EQU 1
disableRom        EQU 0

setThirtyTwo      EQU 1
clearThirtyTwo    EQU 0

maxRomsInPackage  EQU 4
lastIndexToCheck  EQU 6	; This is (maxRomsInPackage - 2) * 2

thirtyTwoKMask    EQU 080H
invalidRom        EQU 0FFH

thirtyTwoKPart    EQU 403H
activateRom       EQU 405H
firstQtr          EQU 440H
secondQtr         EQU 441H
thirdQtr	         EQU 442H
fourthQtr         EQU 443H

endChksumArea     EQU 7FFEH

romsBaseAddr      EQU 08000H
thirdQtrAddr      EQU 09000H
romHeaderAddr     EQU 09FF0H
romInfoSignature  EQU 05242H	; Signature is RB (Rom Buffer or Roast Beef)
romHere	         EQU 0BB66H


RomInfoType   STRUC
	signature     DW ?
	romsExist     DB ?
	curRomSlot    DB ?
	indexes       DB numLogHwSlots DUP (?)
RomInfoType   ENDS

SlotInfoType  STRUC
	validRom      DB ?
	waitStateCode DB ?
	romsID        DW ?
	slots         DW ?
	systemType    DB ?
	romsSize      DB ?
SlotInfoType  ENDS

RomHeaderType STRUC
	romHereFlag   DW ?
	sysIndicator  DB ?
	systemRom     DB ?
	bootSector    DW ?
	romAddr       DB ?
	romSize       DB ?
	numWaitStates DB ?
	dirSelector   DW ?
	dirLength     DW ?
	numFiles      DW ?
	pageZeroSel   DW ?
	numPages      DW ?
	romID         DW ?
	stuff         DB 112 DUP (?)
	numRomsInPkg  DB ?
	romValue      DW   4 DUP (?)
RomHeaderType ENDS
$EJECT

CODE SEGMENT BYTE PUBLIC 'CODE'
	ASSUME CS:CGROUP, DS:NOTHING

; RomSubsystem

; Entry (Assuming that AH = 0E4h has gotten us here)
;  AL = 0: Get ROM subsystem information
;  AL = 1: Initialize ROM subsystem
;  AL = 2: Get ROM image information - Includes old fnc 7 information
;  AL = 3: Bank in a ROM image (& enable ROMs doing so) - Was old fnc 6
;  AL = 4: Disable ROMs - Was old fnc 5

;  See below for specific register needs for each section

; Exit
;  Carry set indicates error
;  AH = 0: No error
;  AH = 1: Invalid function number (in AL)
;  AH = 2: Passes data segment is invalid
;  AH = 3: No rom images exist
;  AH = 4: image number is out of range

;  See below for specific register returns for each section

RomSysJmpTable LABEL WORD
	DW	OFFSET GetRomSysInfo
	DW	OFFSET InitRomSystem
	DW	OFFSET GetRomImageInfo
	DW	OFFSET BankInRomImage
	DW	OFFSET DisableROMs


RomSubsystem PROC NEAR
	CMP	AL, numValidRomFncs
	JB	ValidRomFunctionCall

InvalidFunction LABEL NEAR
	MOV	AH, romErrInvFnc
	JMP	SHORT RomSubsystemReturn

ValidRomFunctionCall:
	PUSH	DI
	PUSH	SI
	MOV	DI, AX
	AND	DI, 0FFH
	SHL	DI, 1
	CALL	CS:RomSysJmpTable[DI]
	POP	SI
	POP	DI

RomSubsystemReturn:
	CMP	AH, 0	; Comparing to zero clears carry (no error)
	JE	RomSubsystemExit

	STC		; Carry indicates error condition

RomSubsystemExit:
	RET
RomSubsystem ENDP
$EJECT

; Entry
;  Nothing special

; Exit
;  AL = number of logical hardware rom slots
;  BX = number of paragraphs needed for data area
;  CL = number of physical hardware rom slots

GetRomSysInfo PROC NEAR
	MOV	BX, numLogHwSlots	; BH set to zero by moving into word
	MOV	AL, SIZE slotInfoType
	MUL	BL
	ADD	AX, SIZE romInfoType
	ADD	AX, 15	; round up to next paragraph
	SHR	AX, 1
	SHR	AX, 1
	SHR	AX, 1
	SHR	AX, 1	; divide by 16 to get number of paragraphs
	XCHG	AX, BX	; AH = ok; AL = numSlots; BX = num paras
	MOV	CL, numPhysHwSlots
	RET
GetRomSysInfo ENDP


; Entry
;  DS => ROM data area

; Exit
;  AH = error response
;  AL = number of logical rom images

; Register usage


InitRomSystem PROC NEAR
	PUSH	ES
	PUSH	BP
	PUSH	DX
	PUSH	CX
	PUSH	BX

	MOV	AL, enableRom
	MOV	DX, activateRom
	OUT	DX, AL	; Enable roms

	MOV	CX, numLogHwSlots
	MOV	AL, SIZE slotInfoType
	MUL	CL
	ADD	AX, SIZE romInfoType
	MOV	CX, AX
	PUSH	DS	; Clear ROM info area
	POP	ES	; which initializes
	XOR	AX, AX	; romsExist to 0 (FALSE)
	MOV	DI, AX	; and validRom for each
	CLD		; ROM to FALSE
	REP	STOSB

	MOV	DS:signature, romInfoSignature
	MOV	DS:curRomSlot, invalidRom

	MOV	AX, romHeaderAddr
	MOV	ES, AX	; Set ES to point to ROM header area

	XOR	BX, BX	; Initialize romSlot index to 0
	MOV	SI, SIZE romInfoType	; Initialize romInfo pointer

NextRomSlotLoop:
	MOV	AX, BX
	MOV	DX, fourthQtr
	OUT	DX, AL	; Select part of rom that contains header

	CMP	ES:romHereFlag, romHere	; If there is a rom in this slot
	JE	ThisRomExists	; then fill in its info
	JMP	NextRomSlot	; else check the next slot

ThisRomExists:
	MOV	AL, 1
	MOV	DS:[SI].validRom, AL	; slotInfo(i).validRom = TRUE

	CMP	ES:numRomsInPkg, AL	; If rom.numRomsInPkg <= 1
	JBE	RomIsNotMadeOfPackages	; then do non-package case

RomIsMadeOfPackages:
	MOV	DS:[SI].waitStateCode, thirtyTwoKMask

	MOV	AX, BX
	MOV	CL, 12
	SHL	AX, CL
	MOV	BP, AX	; Init 4th quarter to this rom slot

	XOR	DI, DI	; Init rom quarter (to search) to 0

NextPackageLoop:
	XOR	CX, CX	; Initialize search rom slot to 0

	CMP	ES:[DI].romValue, CX
	JZ	NextPackage

NextSearchSlotLoop:
	MOV	AX, CX
	MOV	DX, thirdQtr
	OUT	DX, AL

	PUSH	DS	; Save pointer to slot info area
	PUSH	SI

	MOV	SI, thirdQtrAddr	; Set up segment address of
	MOV	DS, SI	; area to be checksummed
	MOV	SI, endChksumArea	; Last word in area to checksum
	XOR	DX, DX	; Accumulator for checksum
	STD		; Do it backwards

ChecksumLoop:
	LODSW	; Get next (prev actually) word
	ADD	DX, AX	; Add to total
	OR	SI, SI	; See if still adding
	JNS	ChecksumLoop	; If so go back

	POP	SI
	POP	DS	; Restore pointer to slot info area

	CMP	ES:[DI].romValue, DX	; If checksum doesn't matches for this slot
	JNE	NextSearchSlot	; Try the next one

	MOV	AX, DI
	SHL	AX, 1
	XCHG	AX, CX
	SHL	AX, CL
	OR	BP, AX
	JMP	SHORT NextPackage

NextSearchSlot:
	INC	CX
	CMP	CX, numLogHwSlots
	JB	NextSearchSlotLoop

	MOV	DS:[SI].validRom, 0	; slotInfo(i).validRom = FALSE

NextPackage:
	INC	DI
	INC	DI
	CMP	DI, lastIndexToCheck
	JB	NextPackageLoop

	MOV	DS:[SI].slots, BP	; slotInfo(i).slots = slots
	JMP	SHORT StoreOtherRomInfo

RomIsNotMadeOfPackages:
	CMP	ES:romSize, 32	; IF rom.romSize > 32 then
	JA	NonPackageSetSlots	; don't set special thirtyTwoKMask

	MOV	DS:[SI].waitStateCode, thirtyTwoKMask

NonPackageSetSlots:
	MOV	AX, BX	; Set all quarters to the same
	MOV	CL, 4	; slot for non-package roms
	SHL	AX, CL
	OR	AX, BX
	MOV	AH, AL
	MOV	DS:[SI].slots, AX	; slotInfo(i).slots = slots

StoreOtherRomInfo:
	MOV	AL, ES:sysIndicator
	MOV	DS:[SI].systemType, AL	; slotInfo(romSlot).systemType = rom.sysType
	MOV	AL, ES:romSize
	MOV	DS:[SI].romsSize, AL	; slotInfo(romSlot).romsSize = rom.romSize
	MOV	AX, ES:romID
	MOV	DS:[SI].romsID, AX	; slotInfo(romSlot).romsID = rom.romID

NextRomSlot:
	INC	BX	; Next rom slot
	ADD	SI, SIZE SlotInfoType
	CMP	BX, numLogHwSlots
	JE	InitRomSystemEnd
	JMP	NextRomSlotLoop

InitRomSystemEnd:
	MOV	AL, disableRom
	MOV	DX, activateRom
	OUT	DX, AL	; Disable roms before exiting

	POP	BX
	POP	CX
	POP	DX
	POP	BP
	POP	ES

	XOR	AX, AX	; Index to logical hardware slot
	MOV	DI, AX	; Index to romInfo.indexes
	MOV	SI, SIZE RomInfoType	; Index to info for next logical hw slot

InitRomSetMapLoop:
	TEST	DS:[SI].validRom, TRUE	; If not a valid rom
	JZ	InitRomSetMapNext	; then try next one

	MOV	DS:[DI].indexes, AL	; Store physical index at logical index
	INC	DI	; Next logical index
	INC	DS:romsExist	; Increment count of logical roms

InitRomSetMapNext:
	INC	AX	; Next logical hardware slot
	ADD	SI, SIZE SlotInfoType	; Next slot's info
	CMP	AL, numLogHwSlots
	JB	InitRomSetMapLoop

	MOV	AL, DS:romsExist	; Return num logical roms to caller
	CALL	VerifyDataBlock	; This will set the error return code

	RET
InitRomSystem ENDP
$EJECT

; Entry
;  DL =  rom image number (or 0FFH for current)
;  DS => ROM data area

; Exit
;  AH = error response
;  AL = logical image rom size (In 32K blocks ?)
;  BX = romsID
;  CX = slot bank info
;  DH = system type
;  DL = rom image number
;  ES = rom base address

GetRomImageInfo PROC NEAR
	CALL	VerifyRomImageNumber
	JNZ	GetRomImageInfoRet

	MOV	AL, DS:[SI].romSize
	MOV  BX, romsBaseAddr
	MOV  ES, BX
	MOV	BX, DS:[SI].romsID
	MOV	CX, DS:[SI].slots
	MOV  DH, DS:[SI].systemType

GetRomImageInfoRet:
	RET
GetRomImageInfo ENDP
$EJECT

; Entry
;  DL =  rom image number (or 0FFH for current)
;  DS => ROM data area

; Exit
;  AH = error response
;  DL = actual rom image banked in
;  AL, BX, CX, DH changed !!

BankInRomImage PROC NEAR
	CALL	VerifyRomImageNumber
	JNZ	BankInRomImageRet

	MOV	DS:curRomSlot, DL	; Set current rom to value in DL

	MOV	AL, enableRom
	MOV	DX, activateRom
	OUT	DX, AL

	MOV	CL, DS:[SI].waitStateCode
	MOV	BX, DS:[SI].slots

	MOV	DX, thirtyTwoKPart	; See if we have a 32K ROM
	MOV	AL, setThirtyTwo
	TEST	CL, thirtyTwoKMask
	JNZ	WriteThirtyTwoKRegister

	MOV	AL, clearThirtyTwo

WriteThirtyTwoKRegister:
	OUT	DX, AL	; set/clear the 32K flag

	MOV	CX, 4	; maximum for roms to a rom image
	MOV	DX, firstQtr

SelectBanksLoop:
	MOV	AL, BL	; get bank values
	AND	AL, 7	; mask all but current slot
	OUT	DX, AL	; select next bank
	INC	DX	; get to next bank port
	SHR	BX, 1
	SHR	BX, 1
	SHR	BX, 1
	SHR	BX, 1	; get to next bank
	LOOP	SelectBanksLoop

	MOV	DL, DS:curRomSlot

BankInRomImageRet:
	RET
BankInRomImage ENDP
$EJECT

; Entry
;  DS => ROM data area

; Exit
;  AH = error response

DisableROMs PROC NEAR
	CALL	VerifyDataBlock
	JNZ	DisableROMsRet	; AH will be 0 if okay

	XCHG	DI, DX	; Save DX in DI
	XCHG	SI, AX	; Save AX in SI
	MOV	AL, disableRom
	MOV	DX, activateRom
	OUT	DX, AL
	XCHG	DX, DI	; Restore DX from DI
	XCHG	AX, SI	; Restore AX from SI

DisableROMsRet:
	RET
DisableROMs ENDP
$EJECT

; Entry
;  DL =  rom image number (or 0FFH for current)
;  DS => ROM data area

; Exit
;  AH = error response (if any)
;  DL = actual rom image number (if no error)
;  SI = index into rom info table

VerifyRomImageNumber PROC NEAR
	CALL	VerifyDataBlock	; Verify that DS is okay and roms exist
	JNZ	VerifyRomImageNumRet	; If any errors exit now

	MOV	DI, DX	; Save DL in case of error
	CMP	DL, 0FFH	; If rom image is explicit
	JNE	RomImageNumberIsGiven	; then check its validity

	MOV	DL, DS:curRomSlot	; Use current rom for DL = 0FFH

RomImageNumberIsGiven:
	MOV	AH, romErrInvImageNum	; Assume invalid image number error
	CMP	DL, DS:romsExist	; If slot number is out of range
	JAE	VerifyRomImageError	; then return error

	MOV	AH, DL	; Save rom image number
	XCHG	AX, SI	; Save AX in SI for a while
	MOV	DI, DX
	AND	DI, 0FFH	; Map logical rom image number
	MOV	DL, DS:[DI].indexes	; to logical hardware slot number
	MOV	AL, SIZE slotInfoType
	MUL	DL	; Index into the rom entry
	ADD	AX, SIZE romInfoType	; Skip past header part
	XCHG	SI, AX	; AX is restored; SI = index
	MOV	DL, AH	; Actual rom image number is restored

	MOV	AH, 0
	JMP	SHORT VerifyRomImageNumRet

VerifyRomImageError:
	MOV	DX, DI	; Restore users DL (stupid!)

VerifyRomImageNumRet:
	OR	AH, AH
	RET
VerifyRomImageNumber ENDP


; Entry
;  DS => ROM data area

; Exit
;  AH = error response

VerifyDataBlock PROC NEAR
	MOV	AH, romErrInvDataBlk	; Assume invalid data block error
	CMP	DS:signature, romInfoSignature ; If signature is not valid
	JNE	VerifyDataBlockRet	; Then return with error

	MOV	AH, romErrNoRomsExist	; Assume no roms exist error
	CMP	DS:romsExist, 0	; If no roms exist
	JE	VerifyDataBlockRet	; Then return with error

	MOV	AH, 0	; Return no error

VerifyDataBlockRet:
	OR	AH, AH	; Re/Set zero flag for callers sake
	RET
VerifyDataBlock ENDP


CODE ENDS

    END
